from uuid import UUID

from flows_sdk.flows import Flow, Manifest, Parameter
from flows_sdk.implementations.idp_v42.additional_blocks import (
    HyperscienceRestApiBlock,
    JSONOutputsBlock,
)
from flows_sdk.package_utils import export_flow
from flows_sdk.utils import str_to_deterministic_uuid_4, workflow_input

ON_ERROR_FLOW_IDENTIFIER = 'ON_ERROR_EXAMPLE_V2'
ON_ERROR_FLOW_UUID = str(str_to_deterministic_uuid_4(ON_ERROR_FLOW_IDENTIFIER))
ON_ERROR_TITLE = 'On Error V2 - flows-sdk example'
ON_ERROR_DESCRIPTION = """
Flow designed to be used as an "on-error" handler for other flows.
\n
With the on-error flow configured, when the original flow fails for any reason, and all
    the configured auto-retries have been exhausted, the on-error flow will be triggered.
\n
The input it receives is the ID of the failed flow run, which can be used to query
    additional information about the failure.
\n
In version V2 of the flow,
    we fetch the failed flow run but omit heavy block input/output payloads
    and then send it to a user-configured output.
    Old versions of the on-error flow use a now-deprecated API endpoint, and will stop
    working in V45 of the product,
    unless updated to use the new lighter endpoint - /api/v5/flow_runs/id/summary.
    The data format of the output conforms to
    the schema documented here: https://docs.hyperscience.ai/#flows-runs and
    here: https://docs.hyperscience.ai/#submissions, under the respective keys
    "flow_run" and "submission".
\n
If you need more complex logic in this flow
    please consult the documentation of Flows SDK (https://flows-sdk.hyperscience.ai)
    and the available public APIs (https://docs.hyperscience.ai)
""".lstrip()


class OnErrorFlowInputKeys:
    failed_run_uuid: str = 'failed_run_uuid'


class OnErrorManifest(Manifest):
    def __init__(self, flow_identifier: str):
        super().__init__(
            identifier=flow_identifier,
            roles=['on_error', 'supporting'],
            input=[
                Parameter(
                    name=OnErrorFlowInputKeys.failed_run_uuid,
                    type='string',
                    title='Failed Flow Run UUID',
                    description='UUID of the failed flow run that triggered this on_error flow',
                    optional=False,
                    ui={'hidden': True},
                ),
            ],
        )


def on_error_flow() -> Flow:
    flow_run_rest_api_block = HyperscienceRestApiBlock(
        reference_name='flow_run_rest_api_block',
        method='GET',
        app_endpoint=f'/api/v5/flow_runs/{workflow_input(OnErrorFlowInputKeys.failed_run_uuid)}/summary',
    )

    json_outputs_block = JSONOutputsBlock(
        inputs={'payload': flow_run_rest_api_block.output('result.data')}
    )

    return Flow(
        uuid=UUID(ON_ERROR_FLOW_UUID),
        owner_email='flows-sdk@hyperscience.ai',
        title=ON_ERROR_TITLE,
        description=ON_ERROR_DESCRIPTION,
        manifest=OnErrorManifest(ON_ERROR_FLOW_IDENTIFIER),
        output={},
        input={},
        blocks=[
            flow_run_rest_api_block,
            json_outputs_block,
        ],
    )


def entry_point_workflow() -> Flow:
    return on_error_flow()


if __name__ == '__main__':
    export_flow(entry_point_workflow())
