import os
from uuid import UUID

from flows_sdk.blocks import Outputs
from flows_sdk.flows import Flow, Manifest, Parameter
from flows_sdk.package_utils import export_flow
from flows_sdk.utils import icon_path_to_base64, str_to_deterministic_uuid_4, workflow_input

BASE_DIR = os.path.dirname(os.path.abspath(__file__))

IDP_SUBMISSION_NOTIFY_IDENTIFIER = 'IDP_SUBMISSION_NOTIFY_EXAMPLE'
IDP_SUBMISSION_NOTIFY_UUID = str(str_to_deterministic_uuid_4(IDP_SUBMISSION_NOTIFY_IDENTIFIER))
IDP_SUBMISSION_NOTIFY_TITLE = 'Submission State Notifications - flows-sdk example'
IDP_OUTPUT_ROLE = 'idp_output'


class SubmissionNotifyInputKeys:
    submission: str = 'submission'


class SubmissionNotifyManifest(Manifest):
    def __init__(self, flow_identifier: str):
        super().__init__(
            identifier=flow_identifier,
            roles=['notifications', 'supporting'],
            input=[
                Parameter(
                    name=SubmissionNotifyInputKeys.submission,
                    type='Submission',
                    title='Submission Object',
                    ui={
                        'hidden': True,
                    },
                )
            ],
            ui={
                'icon': icon_path_to_base64(os.path.join(BASE_DIR, 'submission_notify_icon.png')),
            },
        )


def idp_submission_notify_workflow() -> Flow:
    return Flow(
        title=IDP_SUBMISSION_NOTIFY_TITLE,
        owner_email='flows-sdk@hyperscience.ai',
        description=(
            'Send notifications to external systems when a submission has been '
            'created or is waiting for supervision. \n'
            'If the "Document Processing" flow is live, this flow '
            'must also be live, but it can be empty.'
        ),
        manifest=SubmissionNotifyManifest(IDP_SUBMISSION_NOTIFY_IDENTIFIER),
        blocks=[
            Outputs(
                reference_name='outputs',
                title='Outputs',
                description=(
                    'Send submission data to external systems when a submission '
                    'has been created or is waiting for supervision'
                ),
                role_filter=[IDP_OUTPUT_ROLE],
                input_template={
                    'submission': {'id': workflow_input('submission.id')},
                    'enabled': True,
                },
                blocks=[],
            )
        ],
        uuid=UUID(IDP_SUBMISSION_NOTIFY_UUID),
        input={},
        output={},
    )


def entry_point_workflow() -> Flow:
    return idp_submission_notify_workflow()


if __name__ == '__main__':
    export_flow(entry_point_workflow())
