from uuid import UUID

from flows_sdk.error_handling import ErrorHandling
from flows_sdk.flows import Flow
from flows_sdk.implementations.idp_v42 import idp_blocks, idp_values
from flows_sdk.implementations.idp_v42.idp_blocks import IDPCoreBlock
from flows_sdk.implementations.idp_v42.idp_values import (
    DEFAULT_IDP_CORE_IDENTIFIER,
    DEFAULT_IDP_SUBMISSION_NOTIFY_IDENTIFIER,
    IdpWorkflowConfig,
    IDPWrapperManifest,
    Settings,
    get_idp_wf_config,
)
from flows_sdk.package_utils import export_flow
from flows_sdk.utils import str_to_deterministic_uuid_4

IDP_IDENTIFIER = 'DOCUMENT_PROCESSING_EXAMPLE'
IDP_UUID = str(str_to_deterministic_uuid_4(IDP_IDENTIFIER))
IDP_TITLE = 'Document Processing - flows-sdk example'


def idp_workflow(idp_wf_config: IdpWorkflowConfig) -> Flow:
    # By default, all the notification flows are set to `None`.
    # I.e. the system will send no intermediate notifications for submission change events.
    # So let's specify the notification flows that we want to use explicitly here:
    idp_wf_config.notification_workflow_initialization = DEFAULT_IDP_SUBMISSION_NOTIFY_IDENTIFIER
    idp_wf_config.notification_workflow_classification = DEFAULT_IDP_SUBMISSION_NOTIFY_IDENTIFIER
    idp_wf_config.notification_workflow_identification = DEFAULT_IDP_SUBMISSION_NOTIFY_IDENTIFIER
    idp_wf_config.notification_workflow_transcription = DEFAULT_IDP_SUBMISSION_NOTIFY_IDENTIFIER
    idp_wf_config.notification_workflow_flexible_extraction = (
        DEFAULT_IDP_SUBMISSION_NOTIFY_IDENTIFIER
    )

    idp_flow = IDPCoreBlock(idp_wf_config, identifier=DEFAULT_IDP_CORE_IDENTIFIER)

    outputs = idp_blocks.IDPOutputsBlock(inputs={'submission': idp_flow.output('submission')})

    return Flow(
        uuid=UUID(IDP_UUID),
        owner_email='flows-sdk@hyperscience.ai',
        title=IDP_TITLE,
        description='Use this flow to manage your input and output connector configuration and '
        'document processing workflow settings.',
        manifest=IDPWrapperManifest(IDP_IDENTIFIER),
        output={'submission': idp_flow.output('submission')},
        triggers=idp_values.IDPTriggers(),
        input={
            Settings.LayoutReleaseUuid: None,
        },
        blocks=[
            idp_flow,
            outputs,
        ],
        error_handling=ErrorHandling(on_error_flow=None),
    )


def entry_point_workflow() -> Flow:
    idp_wf_config = get_idp_wf_config()
    return idp_workflow(idp_wf_config)


if __name__ == '__main__':
    export_flow(entry_point_workflow())
